/** @file   shadowtable.h
 * @brief   Declaration of ShadowTable - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_SHADOWTABLE_H
#define H_WWW_SHADOWTABLE_H

#include "rect2d.h"
#include <list>

namespace WeWantWar {

/** @class  ShadowTable
 * @brief   Manages the dynamic shadows.
 * @author  Tomi Lamminsaari
 *
 * The tilemaps have now a certain layer for shadow information. An instance
 * of this class creates the shadows on certain area and is capable of drawing
 * them over the given bitmap.
 *
 * The shadows use allegro's multiply-blending and the shadow blocks are
 * greyscale sprites so it's quite slow operation.
 */
class ShadowTable
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     aArea             The area this shadowtable manages.
   */
	ShadowTable( const eng2d::Rect2D& aArea );


	/** Destructor
   */
	virtual ~ShadowTable();

private:

	/** Copy constructor.
   * @param     rO                Reference to another ShadowTable
   */
  ShadowTable( const ShadowTable& rO );

	/** Assignment operator
   * @param     rO                Reference to another ShadowTable
   * @return    Reference to us.
   */
  ShadowTable& operator = ( const ShadowTable& rO );

public:



  ///
  /// Methods
  /// =======

  /** Applies the shadows on given bitmap
   * @param     aTarget           Pointer to target bitmap.
   */
  virtual void applyShadows( BITMAP* aTarget );

  /** Sets the alpha level we use.
   * @param     aAlpha            The alpha level
   */
  virtual void setAlpha( int aAlpha );
  
  /** Clears the contents of this ShadowTable
   */
  virtual void clear();
  

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the area that is covered by this shadowtable.
   * @return    The area
   */
  eng2d::Rect2D area() const;

  /** Returns the alpha level we use.
   * @return    The alpha level
   */
  int alpha() const;
  

protected:

  /** The shadow tile information is stored in the instances of this
   * structure.
   */
  class ShadowItem {
  public:
    /** The x-coordinate of the shadow tile. */
    int iShadowX;
    /** The y-coordinate of the shadow tile. */
    int iShadowY;
    /** The index number of the shadow tile. */
    int iShadowIndex;
    
    /** Constructor.
     */
    ShadowItem();
    
    /** Constructor.
     * @param   aShadowX        Shadow's X coordinate in tiles.
     * @param   aShadowY        Shadow's Y coordinate in tiles.
     * @param   aShadowIndex    Index number of the shadow tile.
     */
    ShadowItem( int aShadowX, int aShadowY, int aShadowIndex );
  };

  /** Creates the needed shadow blocks for High tree
   * @param   aCenterX          Tree's center X in tiles.
   * @param   aCenterY          Tree's center Y in tiles.
   */
  void createHighTreeShadow( int aCenterX, int aCenterY );
  
  /** Creates the needed shadow blocks for medium altitude tree
   * @param   aCenterX          Tree's center X in tiles.
   * @param   aCenterY          Tree's center Y in tiles.
   */
  void createMediumTreeShadow( int aCenterX, int aCenterY );


  ///
  /// Members
  /// =======

  /** All the shadowtiles are stored in this linked list. */
  std::list< ShadowItem* > iShadows;
  
  /** The area of the map that is covered by this ShadowTable. */
  eng2d::Rect2D iArea;
  
  /** The alpha value. Defines how much the shadows darkens the graphics. */
  int           iAlpha;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: shadowtable.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:27:56+02  lamminsa
 * ShadowItem struct was promoted to to class.
 *
 */
 
